function dist = calculateMetricLDS(system,metricList,metricParam)
% This is the main distance computing function. Use either default metric
% parmeters or provide custom parameters in metricParam
%
% INPUTS
%   system     -    Cell array of system parameters
%   metricList -    Array of metrics to be returned. Choose from the list:
%       Distances:
%
%           Subspace angles based distances
%       1 - Finsler
%       2 - Matrin
%       3 - Gap
%       4 - Frobenius
%       5 - Matrin Kernel to Distance
%           
%           Binet Cauchy Based Distances
%       6 - Trace Kernel to Distance
%       7 - Trace of P - solution of Stlvester eqn - converted to distance
%       8 - Determinant of P - solution of Sylvester eqn - converted to distance
%       9 - Max Singular Value of P - solution of Sylvester eqn - converted to distance
% 
%       Kernels
%
%            Supsace angles based kernels
%       20 - Martin Kernel
%
%            Binet Cauchy kernels
%       21 - Trace Kernel
%       22 - Trace of P - solution of Sylvester eqn
%       23 - Determinant of P - solution of Sylvester eqn
%       24 - Max Singular Value of P - solution of Sylvester eqn
%
%       NOTE: Distances 7,9 and Kernels 22,24 require all systems to be in
%             a canonical form such as the observability/controllability or
%             Jordan canonical form for proper comparison
%
%   metricParam -   Parameter values for the metrics
%            Subspace angles based distances and kernels
%       .isARMA - 0 for AR systems (default), 1 for ARMA
%
%            Binet Cauchy kernels based distances and kernels
%       .Lambda - Dampening factor for the Binet-Cauchy kernels [Default: 0.9]
%       .Cov_U  - Weight factor for noise covariance [Default: 1]
%       .Alpha  - Weighting Heuristic between two parts of the Binet-Cauchy
%                 kernels [Default: 0.5 for no heuristic weighting] 
%
% OUTPUTS
%   dist    -       [N x N] all-pair distance matrix
%
% EXAMPLE
%
%   D = calculateMetricLDS(sysParam,[1 2 3 4 5 20]);
%
%% Written by : Avinash Ravichandran, Rizwan Chaudhry
%% $DATE      : 09-Sep-2011 17:57:43 $
%% $REVISION  : 0.5.0 $

global verbose;

N = length(system);
n = size(system{1}.A,1);
subspaceAngleMetric = [1 2 3 4 5 20];
binetCauchyMetric   = [6 7 8 9 21 22 23 24];

if nargin<3
    metricParam.isARMA = 0;
    
    metricParam.Lambda = 0.9;
    metricParam.Cov_U  = 1;
    metricParam.Alpha  = 0.5;
end

isSubspaceAngle = ~isempty(intersect(metricList,subspaceAngleMetric));
isBinetCauchy   = ~isempty(intersect(metricList,binetCauchyMetric));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculating Subspace Angle if needed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if (isSubspaceAngle)
    if verbose
        fprintf('Calculating Subspace Angles...');
        fprintf('%i/%05i',N,0);
    end

    angles = zeros(size(system{1}.A,1),N,N);
    for i = 1:N
        testSys = system{i};
        if verbose;fprintf('\b\b\b\b\b%05i',i);end
        for j = 1:i % Can be replaced with parfor
            if metricParam.isARMA
                angles(:,i,j) = subspaceAnglesARMA(testSys, system{j});
            else
                angles(:,i,j) = subspaceAnglesAR(testSys, system{j});
            end
        end
    end
end
if verbose;fprintf('\n');end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculating Binet Cauchy Parameters if needed
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if (isBinetCauchy)
    M     = zeros(n,n,N,N);
    sigma = zeros(n,n,N,N);
    tempM = zeros(n,n,N);
    tempsigma = zeros(n,n,N);

    for i = 1:N
        testSys = system{i};
        AK = testSys.A;
        for j = 1:N % Can be replaced with parfor
            tempM(:,:,j) = dlyap(metricParam.Lambda.*AK', system{j}.A, testSys.C' * system{j}.C);
            tempsigma(:,:,j) = system{j}.x0 * testSys.x0';
        end
        M(:,:,i,:) = tempM;
        sigma(:,:,i,:) = tempsigma;
    end
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculating Distances
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
if ~isSubspaceAngle
    angles = 0;
end

if ~isBinetCauchy
    M = 0;
    sigma = 0;
end

dist = zeros(N,N,length(metricList));

if verbose
    disp('Calculating Distances..');
end

parfor metricIndex=1:length(metricList) % Can be replaced with parfor

switch metricList(metricIndex)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Subspace Angle Based distances
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    case 1
        %%% Finsler Distance
        dist(:,:,metricIndex) = calculateFinslerDistance(angles);

    case 2
        %%% Matrin Distance
        dist(:,:,metricIndex) = calculateMartinDistance(angles);

    case 3
        %%% Gap Distance
        dist(:,:,metricIndex) = calculateGapDistance(angles);

    case 4
        %%% Frobenius Distance
        dist(:,:,metricIndex) = calculateFrobeniusDistance(angles);

    case 5
        %%% Matrin Kernel to Distance
        dist(:,:,metricIndex) = calculateMartinKernelToDistance(angles);

    case 20
        %%% Martin Kernel
        dist(:,:,metricIndex) = calculateMartinKernel(angles);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % Binet Cauchy Based Distances
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    case 6
        %%% Trace Kernel to Distance
        ktemp = calculateTraceKernel(M,sigma,system,metricParam);
        dist(:,:,metricIndex) = kernelToDistance(normalizeKernel(ktemp));

    case 7
        %%% Trace of Stlvester eqn converted to distance
        ktemp = calculatePartialTraceKernel(M,sigma,system,metricParam);
        dist(:,:,metricIndex) = kernelToDistance(normalizeKernel(ktemp));

    case 8
        %%% Determinant of Sylvester equation converted to distance
        ktemp = calculatePartialDeterminantKernel(M,sigma,system,metricParam);
        dist(:,:,metricIndex) = kernelToDistance(normalizeKernel(ktemp));

    case 9
        %%% Max Singular Value converted to distance
        ktemp = calculatePartialMaxSVKernel(M,sigma,system,metricParam);
        dist(:,:,metricIndex) = normalizeKernel(kernelToDistance(normalizeKernel(ktemp)));

    case 21
        %%% Trace Kernel
        dist(:,:,metricIndex) = normalizeKernel(calculateTraceKernel(M,sigma,system,metricParam));

    case 22
        %%% Trace of Solution to sylvester equation
        dist(:,:,metricIndex) = normalizeKernel(calculatePartialTraceKernel(M,sigma,system,metricParam));

    case 23
        %%% Determinant of solution of Sylvester Equation
        dist(:,:,metricIndex) = normalizeKernel(calculatePartialDeterminantKernel(M,sigma,system,metricParam));

    case 24
        %%% Max Singular Value
        dist(:,:,metricIndex) = normalizeKernel(calculatePartialMaxSVKernel(M,sigma,system,metricParam));

end

end
