function kernel = calculatePartialMaxSVKernel(varargin)
% Compute the Maximum Singular value kernel between dynamical systems
%
% INPUTS
% Takes variable inputs of the type:
%   option 1 -  (system1, system2) - two system parameters
%   option 2 -  (system1, system2, metricParams) - two system parameters
%                and custom values for metric properties
%   option 3 -  (M,sigma,system,metricParams) - all-pair solution matrix for
%               corresponding sylvester equations, inital state covariance
%               estimates, all system parmaeters and metric parameters
%               metricParams  -   parameters for computing the Binet-Cauchy kernels
%                   .Lambda - Dampening factor for the Binet-Cauchy kernels [Default: 0.9]
%                   .Cov_U  - Weight factor for noise covariance [Default: 1]
%                   .Alpha  - Weighting Heuristic between two parts of the Binet-Cauchy
%                             kernels [Default: 0.5 for no heuristic weighting] 
%
% OUTPUTS
%   kernel  -    kernel between 2 systems or all-pair Partial determinant
%   kernel matrix [N x N]
%
% EXAMPLE
%
%  kernel = calculatePartialMaxSVKernel(sysParamsCellArray);
%
%% Written by : Avinash Ravichandran, Rizwan Chaudhry
%% $DATE      : 09-Sep-2011 17:57:43 $
%% $REVISION  : 0.5.0 $


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialization
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[M,sigma,system,mParam] = preprocessSystems(1,varargin);

Lambda = mParam.Lambda;
Cov_U  = mParam.Cov_U;
Alpha  = mParam.Alpha;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Calculating Partial Max SV Kernel
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%% Calculating the distances from the angle
N      = size(M,3);
kernel = zeros(N,N);

for i = 1:N
    parfor j = 1:N
        kernel(i,j) = max(svd((M(:,:,i,j))));
    end
end

kernel = 0.5*(kernel + kernel');    % Remove very small numerical errors

if N == 1
    kernel = kernel(1,2);
end
