% This code tests the performance of the ORGEN algorithm for solving the
% elastic net optimization problem
% 
% min_x lambda ||x||_1 + (1-lambda)/2 ||x||_2^2 + nu/2 ||b - Ax||_2^2 (*)
% 
%   Chong You, Chun-guang Li, Daniel Robinson, Rene Vidal,
%   "Oracle Based Active Set Algorithm for Scalable Elastic Net Subspace
%   Clustering", CVPR 2016.

% This code generates an A of size D by N, a vector x of length N with
% sparsity s and b = A*x. Two solvers are then used to solve for (*) above.
% Solver 1 is a Homotopy solver. Solver 2 is the ORGEN method. It shows
% that the objective value given by the two solvers are the same but the
% ORGEN is much faster. You can also try other solvers, e.g. the RFSS 
% https://sites.google.com/site/igorcarron2/cs

% Copyright Chong You @ Johns Hopkins University, 2016
% chong.you1987@gmail.com

addpath('Tools')

% data parameter
D = 100;
N = 10^5;
s = 100;
% model parameter
lambda = 0.9;
nu = 100;
% solvers
solver1 = @(A, b) CompSens_EN_Homotopy(A, b, lambda, nu); % homotopy
solver2 = @(A, b) ORGEN(A, b, lambda, nu, 'Nsample', 500, 'maxiter', 50, 'outflag', 0); % ORGEN with homotopy
% Try also other solvers
% @(A, b) rfss( A, b, lambda / nu, (1-lambda) / nu ); % rfss
% @(A, b) CompSens_EN_Homotopy(A, b, lambda, nu); % homotopy
% @(A, b) ORGEN(A, b, lambda, nu, 'Nsample', 500, 'maxiter', 50, 'outflag', 0); % ORGEN with homotopy
% @(A, b) ORGEN(A, b, lambda, nu, 'EN_solver', 'rfss', 'Nsample', 500, 'maxiter', 50, 'outflag', 0); % ORGEN with rfss

Nexperiment = 5;
for ii = 1:Nexperiment
    A = randn(D, N);
%     A = cnormalize_inplace(A);
    
    x = zeros(N, 1);
    x(randperm(N, s)) = randn(s, 1);
    b = A * x;
%     b = cnormalize(b);

    tic
    x1 = solver1(A, b);
    time1 = toc;
    obj1 = lambda * sum(abs(x1)) + (1-lambda)/2 * sum(x1 .^2) + nu/2 * sum((b - A * x1) .^2);

    tic
    x2 = solver2(A, b);
    time2 = toc;
    obj2 = lambda * sum(abs(x2)) + (1-lambda)/2 * sum(x2 .^2) + nu/2 * sum((b - A * x2) .^2);
    
    fprintf('%d-th experiment: obj1 = %f, obj2 = %f, time1 = %f, time2 = %f\n', ii, obj1, obj2, time1, time2)
end





